"""Instantiate each fuel assembly as an OpenMC Lattice."""

import numpy as np

import openmc
# from tensorflow.python.ops.numpy_ops import empty

from nuscale.materials import mats
from nuscale.surfaces import surfs, pin_cell_pitch
from nuscale.pins import (pin_universes)


def make_assembly(name, universes):
    """Instantiate an OpenMC Lattice for this fuel assembly.

    This method creates a 17x17 PWR lattice with axially spaced
    sleeves defined in the global surfs dictionary.

    Parameters
    ----------
    name: str
        The string name to assign to the Lattice
    universes: numpy.ndarray of openmc.Universe
        A 2D NumPy array of Universes to use for the Lattice

    Returns
    -------
    universe: openmc.Universe
        A Universe with a Cell filled by the Lattice
    """

    # Instantiate the lattice
    lattice = openmc.RectLattice(name=name)
    lattice.lower_left = (-17.*pin_cell_pitch/2., -17.*pin_cell_pitch/2.)
    lattice.pitch = (pin_cell_pitch, pin_cell_pitch)
    lattice.universes = universes
    # Fill outer space of the lattice with coolant
    cell = openmc.Cell(name='Outer assembly lattice cell', fill=mats['coolant'])
    outer_universe = openmc.Universe(name='Outer assembly lattice universe')
    outer_universe.add_cell(cell)
    lattice.outer = outer_universe

    # Create rectangular prism for lattice grid box
    # lat_grid_box = surfs['lat grid box outer'] & ~surfs['lat grid box inner']
    # lat_grid_box = -surfs['lat grid box outer'] & +surfs['lat grid box inner']

    # Add lattice to bounding cell
    univ_name = name + ' lattice'
    universe = openmc.Universe(name=univ_name)
    cell = openmc.Cell(name=univ_name)
    cell.fill = lattice
    # cell.region = surfs['lat grid box inner']
    # cell.region = -surfs['lat grid box inner']
    universe.add_cell(cell)

    # # Add outer water cell
    # cell = openmc.Cell(name=univ_name + ' outer water')
    # cell.fill = mats['H2O']
    # # cell.region = ~surfs['lat grid box outer']
    # cell.region = +surfs['lat grid box outer']
    # universe.add_cell(cell)
    #
    # # Make bottom axial cell for outside of assembly (without sleeve)
    # cell = openmc.Cell(name=univ_name + ' axial (0)')
    # cell.fill = mats['H2O']
    # cell.region = lat_grid_box & -surfs['grid1bot']
    # universe.add_cell(cell)
    #
    # # Make axial cell for outside of assembly (with sleeve)
    # cell = openmc.Cell(name=univ_name + ' axial (1)')
    # cell.fill = mats['In']
    # cell.region = lat_grid_box & +surfs['grid1bot'] & -surfs['grid1top']
    # universe.add_cell(cell)
    #
    # # Make axial cell for outside of assembly (without sleeve)
    # cell = openmc.Cell(name=univ_name + ' axial (2)')
    # cell.fill = mats['H2O']
    # cell.region = lat_grid_box & +surfs['grid1top'] & -surfs['grid2bot']
    # universe.add_cell(cell)
    #
    # # Make axial cell for outside of assembly (with sleeve)
    # cell = openmc.Cell(name=univ_name + ' axial (3)')
    # cell.fill = mats['Zr']
    # cell.region = lat_grid_box & +surfs['grid2bot'] & -surfs['grid2top']
    # universe.add_cell(cell)
    #
    # # Make axial cell for outside of assembly (without sleeve)
    # cell = openmc.Cell(name=univ_name + ' axial (4)')
    # cell.fill = mats['H2O']
    # cell.region = lat_grid_box & +surfs['grid2top'] & -surfs['grid3bot']
    # universe.add_cell(cell)
    #
    # # Make axial cell for outside of assembly (with sleeve)
    # cell = openmc.Cell(name=univ_name + ' axial (5)')
    # cell.fill = mats['Zr']
    # cell.region = lat_grid_box & +surfs['grid3bot'] & -surfs['grid3top']
    # universe.add_cell(cell)
    #
    # # Make axial cell for outside of assembly (without sleeve)
    # cell = openmc.Cell(name=univ_name + ' axial (6)')
    # cell.fill = mats['H2O']
    # cell.region = lat_grid_box & +surfs['grid3top'] & -surfs['grid4bot']
    # universe.add_cell(cell)
    #
    # # Make axial cell for outside of assembly (with sleeve)
    # cell = openmc.Cell(name=univ_name + ' axial (7)')
    # cell.fill = mats['Zr']
    # cell.region = lat_grid_box & +surfs['grid4bot'] & -surfs['grid4top']
    # universe.add_cell(cell)
    #
    # # Make top axial cell for outside of assembly (without sleeve)
    # cell = openmc.Cell(name=univ_name + ' axial (8)')
    # cell.fill = mats['H2O']
    # cell.region = lat_grid_box & +surfs['grid4top'] & -surfs['grid5bot']
    # universe.add_cell(cell)
    #
    # # Make axial cell for outside of assembly (with sleeve)
    # cell = openmc.Cell(name=univ_name + ' axial (9)')
    # cell.fill = mats['Zr']
    # cell.region = lat_grid_box & +surfs['grid5bot'] & -surfs['grid5top']
    # universe.add_cell(cell)
    #
    # # Make top axial cell for outside of assembly (without sleeve)
    # cell = openmc.Cell(name=univ_name + ' axial (last)')
    # cell.fill = mats['H2O']
    # cell.region = lat_grid_box & +surfs['grid5top']
    # universe.add_cell(cell)

    return universe


def assembly_universes():
    """Generate universes for SMR fuel assemblies.

    Parameters
    ----------
    ring_radii : iterable of float
        Radii of rings in fuel (note that this doesn't need to include the
        full fuel pin radius)
    num_axial : int
        Number of axial subdivisions in fuel
    depleted : bool
        Whether fuel should contain nuclides as though it were depleted

    Returns
    -------
    dict
        Dictionary mapping a universe name to a openmc.Universe object

    """
    pins = pin_universes(grid=True)

    # Create dictionary to store assembly universes
    univs = {}

    # commonly needed universes
    gtu = pins['GT CR ejected stack']
    gti = pins['GT CR inserted stack']
    gtc = pins['GT empty stack']
    ba = pins['GdC02 stack']

    # Define the NumPy array indices for assembly locations where there may be CR guide tubes
    nonfuel_y = np.array([2,2,2,3,3,5,5,5,5,5,8,8,8,8,8,11,11,11,11,11,13,13,14,14,14])
    nonfuel_x = np.array([5,8,11,3,13,2,5,8,11,14,2,5,8,11,14,2,5,8,11,14,3,13,5,8,11])

    # Define the NumpPy array indices for assembly locations where there may be burnable absorbers
    ba_y = np.array([2,2,3,3,6,6,7,7,9,9,10,10,13,13,14,14])
    ba_x = np.array([2,14,7,9,6,10,3,13,3,13,6,10,7,9,2,14])

    #### A01 Fuel Assembly (no burnable absorbers)
    universes = np.empty((17,17), dtype=openmc.Universe)
    universes[:,:] = pins['A01 stack']
    universes[nonfuel_y, nonfuel_x] = gtu
    universes[8, 8] = gtc
    univs['A01 Assembly no CRs'] = make_assembly('A01 Assembly no CRs', universes)

    #### A02 Fuel Assembly (no burnable absorbers)
    universes = np.empty((17,17), dtype=openmc.Universe)
    universes[:,:] = pins['A02 stack']
    universes[nonfuel_y, nonfuel_x] = gtu
    universes[8, 8] = gtc
    univs['A02 Assembly no CRs'] = make_assembly('A02 Assembly no CRs', universes)

    #### B01 Fuel Assembly (no burnable absorbers)
    universes = np.empty((17,17), dtype=openmc.Universe)
    universes[:,:] = pins['B01 stack']
    universes[nonfuel_y, nonfuel_x] = gtu
    universes[8, 8] = gtc
    univs['B01 Assembly no CRs'] = make_assembly('B01 Assembly no CRs', universes)

    #### B02 Fuel Assembly (no burnable absorbers)
    universes = np.empty((17,17), dtype=openmc.Universe)
    universes[:,:] = pins['B02 stack']
    universes[nonfuel_y, nonfuel_x] = gtu
    universes[8, 8] = gtc
    univs['B02 Assembly no CRs'] = make_assembly('B02 Assembly no CRs', universes)

    #### C01 Fuel Assembly (no burnable absorbers)
    universes = np.empty((17,17), dtype=openmc.Universe)
    universes[:,:] = pins['C01 stack']
    universes[nonfuel_y, nonfuel_x] = gtu
    universes[8, 8] = gtc
    univs['C01 Assembly no CRs'] = make_assembly('C01 Assembly no CRs', universes)

    #### C02 Fuel Assembly with burnable absorbers
    universes = np.empty((17,17), dtype=openmc.Universe)
    universes[:,:] = pins['C02 stack']
    universes[nonfuel_y, nonfuel_x] = gtu
    universes[8, 8] = gtc
    universes[ba_y, ba_x] = ba
    univs['C02 Assembly no CRs'] = make_assembly('C02 Assembly no CRs', universes)

    #### C03 Fuel Assembly (no burnable absorbers)
    universes = np.empty((17,17), dtype=openmc.Universe)
    universes[:,:] = pins['C03 stack']
    universes[nonfuel_y, nonfuel_x] = gtu
    universes[8, 8] = gtc
    univs['C03 Assembly no CRs'] = make_assembly('C03 Assembly no CRs', universes)

    #### A01 Fuel Assembly with inserted control rods
    universes = np.empty((17,17), dtype=openmc.Universe)
    universes[:,:] = pins['A01 stack']
    universes[nonfuel_y, nonfuel_x] = gti
    universes[8, 8] = gtc
    univs['A01 Assembly with CRs'] = make_assembly('A01 Assembly with CRs', universes)

    #### B02 Fuel Assembly with inserted control rods
    universes = np.empty((17,17), dtype=openmc.Universe)
    universes[:,:] = pins['B02 stack']
    universes[nonfuel_y, nonfuel_x] = gti
    universes[8, 8] = gtc
    univs['B02 Assembly with CRs'] = make_assembly('B02 Assembly with CRs', universes)

    #### B01 Fuel Assembly with inserted control rods
    universes = np.empty((17,17), dtype=openmc.Universe)
    universes[:,:] = pins['B01 stack']
    universes[nonfuel_y, nonfuel_x] = gti
    universes[8,8] = gtc
    univs['B01 Assembly with CRs'] = make_assembly('B01 Assembly with CRs', universes)

    return univs